'use strict';

/* --------------------------------------------------------------
 daterangepicker.js 2016-04-28
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Date Range Picker
 *
 * Creates an instance of the jQuery UI Daterangepicker widget which enables the user to select
 * a custom date range in the same datepicker, something that is not supported by jQuery UI.
 *
 * This widget requires the "general" translation section in order to translate the day
 * and month names.
 *
 * ### Options
 *
 * You can provide all the options of the following site as data attributes:
 *
 * {@link http://tamble.github.io/jquery-ui-daterangepicker/#options}
 *
 * ### Example
 *
 * ```html
 * <input type="text" data-jse-widget="daterangepicker" data-daterangepicker-date-format="dd.mm.yy" />
 * ```
 *
 * {@link https://github.com/tamble/jquery-ui-daterangepicker}
 *
 * @module JSE/Widgets/datarangepicker
 * @requires jQueryUI-Daterangepicker
 */
jse.widgets.module('daterangepicker', [jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.css', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.js', jse.source + '/vendor/momentjs/moment.min.js', jse.source + '/vendor/jquery-ui-daterangepicker/jquery.comiseo.daterangepicker.min.css'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Escape Key Code
     *
     * @type {Number}
     */

    var ESC_KEY_CODE = 27;

    /**
     * Tab Key Code
     *
     * @type {Number}
     */
    var TAB_KEY_CODE = 9;

    /**
     * Module Selector
     *
     * @type {Object}
     */
    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        presetRanges: [],
        dateFormat: jse.core.config.get('languageCode') === 'de' ? 'dd.mm.yy' : 'mm.dd.yy',
        momentFormat: jse.core.config.get('languageCode') === 'de' ? 'DD.MM.YY' : 'MM.DD.YY',
        applyButtonText: jse.core.lang.translate('apply', 'buttons'),
        cancelButtonText: jse.core.lang.translate('close', 'buttons'),
        datepickerOptions: {
            numberOfMonths: 2,
            changeMonth: true,
            changeYear: true,
            maxDate: null,
            minDate: new Date(1970, 1, 1),
            dayNamesMin: [jse.core.lang.translate('_SUNDAY_SHORT', 'general'), jse.core.lang.translate('_MONDAY_SHORT', 'general'), jse.core.lang.translate('_TUESDAY_SHORT', 'general'), jse.core.lang.translate('_WEDNESDAY_SHORT', 'general'), jse.core.lang.translate('_THURSDAY_SHORT', 'general'), jse.core.lang.translate('_FRIDAY_SHORT', 'general'), jse.core.lang.translate('_SATURDAY_SHORT', 'general')],
            monthNamesShort: [jse.core.lang.translate('_JANUARY_SHORT', 'general'), jse.core.lang.translate('_FEBRUARY_SHORT', 'general'), jse.core.lang.translate('_MARCH_SHORT', 'general'), jse.core.lang.translate('_APRIL_SHORT', 'general'), jse.core.lang.translate('_MAY_SHORT', 'general'), jse.core.lang.translate('_JUNE_SHORT', 'general'), jse.core.lang.translate('_JULY_SHORT', 'general'), jse.core.lang.translate('_AUGUST_SHORT', 'general'), jse.core.lang.translate('_SEPTEMBER_SHORT', 'general'), jse.core.lang.translate('_OCTOBER_SHORT', 'general'), jse.core.lang.translate('_NOVEMBER_SHORT', 'general'), jse.core.lang.translate('_DECEMBER_SHORT', 'general')],
            monthNames: [jse.core.lang.translate('_JANUARY', 'general'), jse.core.lang.translate('_FEBRUARY', 'general'), jse.core.lang.translate('_MARCH', 'general'), jse.core.lang.translate('_APRIL', 'general'), jse.core.lang.translate('_MAY', 'general'), jse.core.lang.translate('_JUNE', 'general'), jse.core.lang.translate('_JULY', 'general'), jse.core.lang.translate('_AUGUST', 'general'), jse.core.lang.translate('_SEPTEMBER', 'general'), jse.core.lang.translate('_OCTOBER', 'general'), jse.core.lang.translate('_NOVEMBER', 'general'), jse.core.lang.translate('_DECEMBER', 'general')]
        },
        onChange: function onChange() {
            var range = $this.siblings('.daterangepicker-helper').daterangepicker('getRange'),
                start = moment(range.start).format(defaults.momentFormat),
                end = moment(range.end).format(defaults.momentFormat),
                value = start !== end ? start + ' - ' + end : '' + start;
            $this.val(value);
        },
        onClose: function onClose() {
            if ($this.val() === '') {
                $this.siblings('i').fadeIn();
            }
        }
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Update the range of the daterangepicker instance.
     *
     * Moment JS will try to parse the date string and will provide a value even if user's value is not
     * a complete date.
     */
    function _updateDaterangepicker() {
        try {
            if ($this.val() === '') {
                return;
            }

            var val = $this.val().split('-');
            var range = {};

            if (val.length === 1) {
                // Single date was selected. 
                range.start = range.end = moment(val[0], options.momentFormat).toDate();
            } else {
                // Date range was selected.
                range.start = moment(val[0], options.momentFormat).toDate();
                range.end = moment(val[1], options.momentFormat).toDate();
            }

            $this.siblings('.daterangepicker-helper').daterangepicker('setRange', range);
        } catch (error) {
            // Could not parse the date, do not update the input value.
            jse.core.debug.error('Daterangepicker Update Error:', error);
        }
    }

    /**
     * On Input Click/Focus Event
     *
     * Display the daterangepicker modal.
     */
    function _onInputClick() {
        if (!$('.comiseo-daterangepicker').is(':visible')) {
            $this.siblings('.daterangepicker-helper').daterangepicker('open');
            $this.siblings('i').fadeOut();
            $(document).trigger('click.sumo'); // Sumo Select compatibility for table-filter rows. 
        }
    }

    /**
     * On Input Key Down
     *
     * If the use presses the escape or tab key, close the daterangepicker modal. Otherwise if the user
     * presses the enter then the current value needs to be applied to daterangepicker.
     *
     * @param {Object} event
     */
    function _onInputKeyDown(event) {
        if (event.which === ESC_KEY_CODE || event.which === TAB_KEY_CODE) {
            // Close the daterangepicker modal. 
            $this.siblings('.daterangepicker-helper').daterangepicker('close');
            $this.blur();
        }
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Daterangepicker needs to be loaded after jquery UI.
        var dependencies = [jse.source + '/vendor/jquery-ui-daterangepicker/jquery.comiseo.daterangepicker.min.js'];

        window.require(dependencies, function () {
            $this.wrap('<div class="daterangepicker-wrapper"></div>').parent().append('<i class="fa fa-calendar"></i>').append('<input type="text" class="daterangepicker-helper hidden" />').find('.daterangepicker-helper').daterangepicker(options);

            $this.siblings('button').css({
                visibility: 'hidden', // Hide the auto-generated button. 
                position: 'absolute' // Remove it from the normal flow.
            });

            $this.on('click, focus', _onInputClick).on('keydown', _onInputKeyDown).on('change', _updateDaterangepicker);

            done();
        });
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
